# DESCRIPTION: RUN THIS SCRIPT TO EXTRACT THE RELEVANT DATA FROM THE XLSX FILES
# WILCOXON TEST FOR CALCIUM DATA
# SAVES THE CLEANED DATA
rm(list=ls(all=TRUE)) # clean work space, common practice      


# load packages
library(openxlsx)
library(data.table)
library(ggpubr)
library(tidyverse)
library(rstatix)

#load Ca2+ response data
data.res <- read.xlsx("Ca_response_cal.xlsx", sheet = 3) %>% 
  filter(opsin != "Antho2aE292A")

stat.test.res <- data.res %>%
  t_test(Ca_res_relWT ~ opsin) %>%
  add_significance()

stat.test.res <- stat.test.res %>% add_xy_position(x = "opsin")

df.summary.res <- data.res %>%
  group_by(opsin) %>%
  summarise(
    sd = sd(Ca_res_relWT, na.rm = TRUE),
    Ca_res_relWT = mean(Ca_res_relWT)
  )

font_sizes <- data.frame("strip_text" = 10,
                         "legend_title" = 10,
                         "axis_title" = 10, "axis_text" = 10,
                         "legend_text" = 10)

barplt_res <- ggplot(data.res, aes(x = fct_rev(fct_reorder(opsin, Ca_res_relWT)), y = Ca_res_relWT)) + 
  geom_col(data = df.summary.res, aes(fill = opsin), width = 0.5) + 
  geom_jitter(position = position_jitter(0), color = "black") + 
  geom_errorbar(aes(ymin = Ca_res_relWT-sd, ymax = Ca_res_relWT+sd), 
                data = df.summary.res, width = 0.2) +
  ylab("Normalised rel. luminescence (WT = 1)") +
  xlab("Antho2a") +
  scale_fill_manual(values = c("#CC0000", "#888888")) +
  scale_x_discrete(labels = c('WT','E292A')) +
  theme_bw() + 
  theme(panel.grid.major = element_blank(), 
        panel.grid.minor = element_blank(),
        plot.title = element_text(hjust = 0.5, size = font_sizes$title,
                                  margin = margin(0,0,20,0)),
        strip.text = element_text(size = font_sizes$strip_text),
        axis.title.x = element_text(size = font_sizes$axis_title,
                                    margin = margin(8,0,0.0)),
        axis.title.y = element_text(size = font_sizes$axis_title,
                                    margin = margin(0,8,0.0)),
        axis.text = element_text(size = font_sizes$axis_text),
        legend.position = "none",
        strip.background = element_rect(fill="darkgrey"),
        plot.margin = unit(c(0.1, 0.1, 0.1, 0.25), "cm")) +
  geom_hline(yintercept=0, linewidth = 0.5)

barplt_res <- barplt_res + 
  stat_pvalue_manual(stat.test.res, tip.length = 0) +
  labs(subtitle = get_test_label(stat.test.res, detailed = TRUE))

ggsave("./figures/Ca_res_norm.pdf", plot = barplt_res,
       width = 4, height = 4, dpi = 600, units = "in")   

#load relative expression
data.exp <- read.xlsx("Ca_response_cal.xlsx", sheet = 2)

df.summary.exp <- data.exp %>%
  group_by(opsin) %>%
  summarise(
    sd = sd(exp, na.rm = TRUE),
    exp = mean(exp)
  )

barplt_exp <- ggplot(data.exp, aes(x = fct_rev(fct_reorder(opsin, exp)), y = exp)) + 
  geom_col(data = df.summary.exp, aes(fill = opsin), width = 0.5) + 
  geom_jitter(position = position_jitter(0), color = "black") + 
  geom_errorbar(aes(ymin = exp-sd, ymax = exp+sd), 
                data = df.summary.exp, width = 0.2) +
  ylab("Relative expression (WT = 1)") +
  xlab("Antho2a") +
  scale_fill_manual(values = c("#CC0000", "#888888")) +
  scale_x_discrete(labels = c('WT','E292A')) +
  theme_bw() + 
  theme(panel.grid.major = element_blank(), 
        panel.grid.minor = element_blank(),
        plot.title = element_text(hjust = 0.5, size = font_sizes$title,
                                  margin = margin(0,0,20,0)),
        strip.text = element_text(size = font_sizes$strip_text),
        axis.title.x = element_text(size = font_sizes$axis_title,
                                    margin = margin(8,0,0.0)),
        axis.title.y = element_text(size = font_sizes$axis_title,
                                    margin = margin(0,8,0.0)),
        axis.text = element_text(size = font_sizes$axis_text),
        legend.position = "none",
        strip.background = element_rect(fill="darkgrey"),
        plot.margin = unit(c(0.1, 0.1, 0.1, 0.25), "cm")) +
  geom_hline(yintercept=0, linewidth = 0.5) +
  geom_hline(yintercept=1, linetype = "dashed", linewidth = 0.5)
  

ggsave("./figures/rel_exp.pdf", plot = barplt_exp,
       width = 4, height = 4, dpi = 600, units = "in")   

